/*
 * To the extent possible under law, Red Hat, Inc. has dedicated all 
 * copyright to this software to the public domain worldwide, pursuant 
 * to the CC0 Public Domain Dedication. This software is distributed 
 * without any warranty.  
 * 
 * See <http://creativecommons.org/publicdomain/zero/1.0/>.
 */

/**
 * purgeResourceOperationHistory.js [<num of days old>]
 * 
 * This JBoss ON CLI script will purge resource operation history from 
 * the JBoss ON system. 
 * 
 * By default, any operations that finished executing more then a year 
 * ago will be purged. This can not be undone.
 * 
 * Optionally, <num of days old> can be passed to the script which will 
 * cause the script to purge any operation that finished executing more 
 * then <num of days old> days ago.
 * 
 * This script is provided as an example of how to purge all resource 
 * operation reports which are older then a certain number of days.
 * 
 * See Red Hat Knowledge Solution 892903 - <https://access.redhat.com/site/solutions/892903>.
 */

function displayUsage() {
    println("Usage: purgeResourceOperationHistory [<num of days old>] [purge] ");
    println("");
    println("<num of days old>  Specifies how many days old a resource operation");
    println("                   history report must be for it to be considered  ");
    println("                   for purge. The default value is 366 if not      ");
    println("                   specified.                                      ");
    println("");
    println("          purge    If provided, this script will actually perform  ");
    println("                   the purge. Without it, the script only tells you");
    println("                   how many operation history reports would be     ");
    println("                   purged in the event the --purge argument was    ");
    println("                   given. This is to prevent accidental execution  ");
    println("                   of the script.                                  ");
    println("");
    println("          help     Display this help message.                      ");
    println("");
}

// by default, only purge operation history older then a year.
var numDaysToKeep = 366;
// this is for safety so we don't accidentally purge unless we mean it
var performPurge = false;
// JavaScript and Rhino do not handle graceful ends/exits
var canContinue = true;
// used for argument parsing
var argPos = 0;

// parse the arguments
while (argPos < args.length) {
    // display help page
    if ("help".equals(args[argPos])) {
        performPurge = true;
        displayUsage();
        canContinue = false;
        break;
    }
    // actually enable the purge
    if ("purge".equals(args[argPos])) {
        performPurge = true;
        argPos++;
        continue;
    }
    // if the argument is a number, its the number of days to keep
    if (!isNaN(args[argPos])) {
        numDaysToKeep = args[argPos];
        argPos++;
        continue;
    }
    // if we get here, we have an invalid argument
    canContinue = false;
    println("Invalid argument: " + args[argPos]);
    println("");
    displayUsage();
    break;
}

// make certain we have a number to work with
if (isNaN(numDaysToKeep)) {
    println("Invalid <older than> number of days argument.");
    println("You must specify a valid value.");
    canContinue = false;
}

// only go into this if argument parsing worked
if (canContinue) {
    // calculate today minus number of days to keep for operation end date
    var cal = java.util.Calendar.getInstance();
    cal.add(java.util.Calendar.DATE, (numDaysToKeep * -1));
    var endTime = cal.getTime();

    /*
     * page is used to keep track of our pageControl location
     * hasMore is used to tell use that we have more pages to pull
     * count is used to keep track of where we are in our total result set
     */
    var page = 0;
    var hasMore = true;
    var count = 0;
    
    var opHistCrit = new ResourceOperationHistoryCriteria();
    opHistCrit.addFilterEndTime(endTime.getTime());
    
    // opHistIds is used to hold the list of resource operation history entries we will delete
    var opHistIds = [];

    // this loop just gets us a list of operation history ids that should be deleted
    while (hasMore) {
        // pull 1000 entries at a time
        opHistCrit.setPageControl(new PageControl(page++, 1000));
        var opHistList = OperationManager.findResourceOperationHistoriesByCriteria(opHistCrit);
        var expectedCount = opHistList.getTotalSize()
        var itr = opHistList.iterator();
        while (itr.hasNext()) {
            count++;
            opHistory = itr.next();
            opHistIds.push(opHistory.getId());
        }
        if (count >= expectedCount) {
            hasMore = false;
        }
    }

    // if we were told to purge, do it -- otherwise just show what we would do
    if (performPurge) {
        println("");
        println("Purging operation history older then " + endTime.toString() + ".");
        // this loop does the real work of the actual deletion
        for (var i = 0; i < opHistIds.length; i++) {
            print("\rRemoving " + (i+1) + " of " + opHistIds.length + " (" + java.lang.Integer(((i+1) / opHistIds.length) * 100) + "%)  ");
            OperationManager.deleteOperationHistory(opHistIds[i], false);
        }
        println("");
        println("Removed " + opHistIds.length + " operation history entries.");
    } else {
        println("");
        println("Would purge " + opHistIds.length + " operation history reports older then ");
        println("    " + endTime.toString() + ".");
        println("");
        println("If you really want to perform the purge, re-run the same command ");
        println("with the purge command-line argument.");
    }
}
