package org.jboss.refarch.eap6.cluster;

import java.io.FileReader;
import java.io.IOException;
import java.util.List;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.jboss.as.cli.CommandFormatException;
import org.jboss.as.cli.CommandLineException;
import org.jboss.refarch.eap6.cli.Attribute;
import org.jboss.refarch.eap6.cli.Client;
import org.jboss.refarch.eap6.cli.Resource;
import org.jboss.refarch.eap6.cli.UnsuccessfulCommandException;

public class Configuration
{

	private final Pattern IP_ADDRESS_PATTERN = Pattern.compile( "(.*?)(\\d+)\\.(\\d+)\\.(\\d+)\\.(\\d+)(.*)" );
	private Client client;
	private String domainName;
	private int offsetUnit;
	private String postgresDriverName;
	private String postgresDriverLocation;
	private String postgresUsername;
	private String postgresPassword;
	private String connectionUrl;
	private String clusterApp;
	private String modClusterProxy;

	public static void main(String[] args) throws CommandLineException, IOException
	{
		String propertyFile = "./configuration.properties";
		if( args.length == 1 )
		{
			propertyFile = args[0];
		}
		Properties props = new Properties();
		props.load( new FileReader( propertyFile ) );
		System.out.println( "properties loaded as: " + props );
		Configuration configuration = new Configuration( props );
		configuration.configure();
		System.out.println( "Done!" );
	}

	public Configuration(Properties properties) throws CommandLineException
	{
		String username = properties.getProperty( "username" );
		String password = properties.getProperty( "password" );
		String domainController = properties.getProperty( "domainController" );
		client = new Client( username, password.toCharArray(), domainController, 9999 );
		domainName = properties.getProperty( "domainName" );
		offsetUnit = Integer.parseInt( properties.getProperty( "offsetUnit" ) );
		postgresDriverName = properties.getProperty( "postgresDriverName" );
		postgresDriverLocation = properties.getProperty( "postgresDriverLocation" );
		postgresUsername = properties.getProperty( "postgresUsername" );
		postgresPassword = properties.getProperty( "postgresPassword" );
		connectionUrl = properties.getProperty( "connectionUrl" );
		clusterApp = properties.getProperty( "deployableApp" );
		modClusterProxy = properties.getProperty( "modClusterProxy" );
	}

	private void configure() throws CommandLineException, IOException
	{
		List<Resource> serverGroups = client.getResourcesByType( null, "server-group" );
		List<Resource> hosts = client.getResourcesByType( null, "host" );

		stopAllServers( serverGroups );
		removeServerConfigs( hosts );
		removeServerGroups( serverGroups );

		updateProfile();
		updateSocketBindingGroups();

		createServerGroups();
		createServerConfigs( hosts );
		client.deploy( postgresDriverLocation + "/" + postgresDriverName );
		setupDataSource();
		client.deploy( clusterApp );

		startAllServers();

		client.disconnect();
		System.out.println( "Disconnected" );
	}

	private void updateProfile() throws IOException, CommandFormatException
	{
		Resource profile = new Resource( "profile", "full-ha" );
		client.load( profile );

		removeExistingProfiles();
		setupModCluster( profile );
		setupMessaging( profile );

		Resource fullHA1 = profile;
		fullHA1.setName( "full-ha-1" );
		client.create( fullHA1 );

		Resource fullHA2 = profile;
		fullHA2.setName( "full-ha-2" );
		configureSecondaryProfile( fullHA2 );
		client.create( fullHA2 );

		Resource fullHA3 = profile;
		fullHA3.setName( "full-ha-3" );
		configureTertiaryProfile( fullHA3 );
		client.create( fullHA3 );
	}

	private void removeExistingProfiles() throws IOException
	{
		List<Resource> profiles = client.getResourcesByType( null, "profile" );
		for( Resource profile : profiles )
		{
			client.load( profile );
			client.remove( profile );
		}
	}

	private void setupModCluster(Resource profile) throws IOException
	{
		Resource modcluster = profile.getChild( "subsystem", "modcluster" );
		Resource configuration = modcluster.getChild( "mod-cluster-config", "configuration" );
		configuration.setAttribute( "advertise", false );
		configuration.setAttribute( "proxy-list", modClusterProxy );
	}

	private void configureSecondaryProfile(Resource profile)
	{
		Resource messaging = profile.getChild( "subsystem", "messaging" );
		messaging.getChild( "hornetq-server", "default" ).setAttribute( "backup-group-name", domainName + "-backup-group-2" );
		messaging.getChild( "hornetq-server", "backup-server-a" ).setAttribute( "backup-group-name", domainName + "-backup-group-3" );
		messaging.getChild( "hornetq-server", "backup-server-b" ).setAttribute( "backup-group-name", domainName + "-backup-group-1" );
		
		profile.getChild( "subsystem", "transactions" ).setAttribute( "node-identifier", 2 );
	}

	private void configureTertiaryProfile(Resource profile)
	{
		Resource messaging = profile.getChild( "subsystem", "messaging" );
		messaging.getChild( "hornetq-server", "default" ).setAttribute( "backup-group-name", domainName + "-backup-group-3" );
		messaging.getChild( "hornetq-server", "backup-server-a" ).setAttribute( "backup-group-name", domainName + "-backup-group-1" );
		messaging.getChild( "hornetq-server", "backup-server-b" ).setAttribute( "backup-group-name", domainName + "-backup-group-2" );
		
		profile.getChild( "subsystem", "transactions" ).setAttribute( "node-identifier", 3 );
	}

	private void setupMessaging(Resource profile) throws IOException
	{
		Resource socketBindingGroup = new Resource( "socket-binding-group", "full-ha-sockets" );
		Resource messagingA = new Resource( "socket-binding", "messaginga" );
		messagingA.setParent( socketBindingGroup );
		messagingA.addAttribute( new Attribute( "port", 5446 ) );
		client.create( messagingA );
		Resource messagingB = new Resource( "socket-binding", "messagingb" );
		messagingB.setParent( socketBindingGroup );
		messagingB.addAttribute( new Attribute( "port", 5447 ) );
		client.create( messagingB );

		Resource messaging = profile.getChild( "subsystem", "messaging" );
		Resource hornetQServer = messaging.getChild( "hornetq-server", "default" );
		hornetQServer.removeAttribute( "clustered" ); //deprecated
		hornetQServer.setAttribute( "shared-store", false );
		hornetQServer.setAttribute( "cluster-password", "clusterPassword1!" );
		hornetQServer.setAttribute( "backup", false );
		hornetQServer.setAttribute( "allow-failback", true );
		hornetQServer.setAttribute( "failover-on-shutdown", false );
		hornetQServer.setAttribute( "check-for-live-server", true );
		hornetQServer.setAttribute( "backup-group-name", domainName + "-backup-group-1" );
		hornetQServer.getChild( "in-vm-connector", "in-vm" ).setAttribute( "server-id", 1 );
		hornetQServer.getChild( "in-vm-acceptor", "in-vm" ).setAttribute( "server-id", 1 );
		Resource pooledCF = hornetQServer.getChild( "pooled-connection-factory", "hornetq-ra" );
		pooledCF.setAttribute( "ha", true );
		pooledCF.setAttribute( "block-on-acknowledge", true );
		pooledCF.setAttribute( "retry-interval", 1000 );
		pooledCF.setAttribute( "retry-interval-multiplier", 1 );
		pooledCF.setAttribute( "reconnect-attempts", -1 );

		Resource hornetQBackupA = Resource.deepCopy( "hornetq-server", "backup-server-a", hornetQServer, messaging );
		hornetQBackupA.setAttribute( "backup", true );
		hornetQBackupA.setAttribute( "backup-group-name", domainName + "-backup-group-2" );
		hornetQBackupA.addChildResource( new Resource( "path", "paging-directory" ) );
		hornetQBackupA.getChild( "path", "paging-directory" ).addAttribute( new Attribute( "path", "messagingpagingbackupa" ) );
		hornetQBackupA.addChildResource( new Resource( "path", "bindings-directory" ) );
		hornetQBackupA.getChild( "path", "bindings-directory" ).addAttribute( new Attribute( "path", "messagingbindingsbackupa" ) );
		hornetQBackupA.addChildResource( new Resource( "path", "journal-directory" ) );
		hornetQBackupA.getChild( "path", "journal-directory" ).addAttribute( new Attribute( "path", "messagingjournalbackupa" ) );
		hornetQBackupA.addChildResource( new Resource( "path", "large-messages-directory" ) );
		hornetQBackupA.getChild( "path", "large-messages-directory" ).addAttribute( new Attribute( "path", "messaginglargemessagesbackupa" ) );
		hornetQBackupA.getChild( "in-vm-connector", "in-vm" ).setAttribute( "server-id", 2 );
		hornetQBackupA.removeChildResource( "remote-connector", "netty-throughput" );
		hornetQBackupA.getChild( "remote-connector", "netty" ).setAttribute( "socket-binding", "messaginga" );
		hornetQBackupA.getChild( "in-vm-acceptor", "in-vm" ).setAttribute( "server-id", 2 );
		hornetQBackupA.removeChildResource( "remote-acceptor", "netty-throughput" );
		hornetQBackupA.getChild( "remote-acceptor", "netty" ).setAttribute( "socket-binding", "messaginga" );
		hornetQBackupA.removeChildResource( "security-setting", "#" );
		hornetQBackupA.removeChildResource( "connection-factory", "RemoteConnectionFactory" );
		hornetQBackupA.removeChildResource( "connection-factory", "InVmConnectionFactory" );
		hornetQBackupA.removeChildResource( "pooled-connection-factory", "hornetq-ra" );

		Resource hornetQBackupB = Resource.deepCopy( "hornetq-server", "backup-server-b", hornetQBackupA, messaging );
		hornetQBackupB.setAttribute( "backup-group-name", domainName + "-backup-group-3" );
		hornetQBackupB.getChild( "path", "paging-directory" ).setAttribute( "path", "messagingpagingbackupb" );
		hornetQBackupB.getChild( "path", "bindings-directory" ).setAttribute( "path", "messagingbindingsbackupb" );
		hornetQBackupB.getChild( "path", "journal-directory" ).setAttribute( "path", "messagingjournalbackupb" );
		hornetQBackupB.getChild( "path", "large-messages-directory" ).setAttribute( "path", "messaginglargemessagesbackupb" );
		hornetQBackupB.getChild( "in-vm-connector", "in-vm" ).setAttribute( "server-id", 3 );
		hornetQBackupB.getChild( "remote-connector", "netty" ).setAttribute( "socket-binding", "messagingb" );
		hornetQBackupB.getChild( "in-vm-acceptor", "in-vm" ).setAttribute( "server-id", 3 );
		hornetQBackupB.getChild( "remote-acceptor", "netty" ).setAttribute( "socket-binding", "messagingb" );
	}

	private void setupDataSource() throws IOException
	{
		Resource dataSources = new Resource( "subsystem", "datasources" );
		Resource clusterDS = new Resource( "data-source", "ClusterDS" );
		dataSources.addChildResource( clusterDS );
		clusterDS.addAttribute( new Attribute( "enabled", true ) );
		clusterDS.addAttribute( new Attribute( "jndi-name", "java:jboss/datasources/ClusterDS" ) );
		clusterDS.addAttribute( new Attribute( "connection-url", connectionUrl ) );
		clusterDS.addAttribute( new Attribute( "driver-class", "org.postgresql.Driver" ) );
		clusterDS.addAttribute( new Attribute( "driver-name", postgresDriverName ) );
		clusterDS.addAttribute( new Attribute( "user-name", postgresUsername ) );
		clusterDS.addAttribute( new Attribute( "password", postgresPassword ) );
		dataSources.setParent( new Resource( "profile", "full-ha-1" ) );
		client.create( clusterDS );
		dataSources.setParent( new Resource( "profile", "full-ha-2" ) );
		client.create( clusterDS );
		dataSources.setParent( new Resource( "profile", "full-ha-3" ) );
		client.create( clusterDS );
	}

	private void updateSocketBindingGroups() throws IOException
	{
		if( offsetUnit > 0 )
		{
			Resource socketBindingGroup = new Resource( "socket-binding-group", "full-ha-sockets" );
			{
				Resource jgroupsMPing = new Resource( "socket-binding", "jgroups-mping" );
				socketBindingGroup.addChildResource( jgroupsMPing );
				Attribute multicast_address = client.readAttribute( jgroupsMPing, "multicast-address" );
				String newIPAddress = modifyIPAddress( multicast_address.getValue().asString(), offsetUnit );
				multicast_address.setValue( newIPAddress );
				client.writeAttribute( jgroupsMPing, multicast_address );
			}
			{
				Resource jgroupsUDP = new Resource( "socket-binding", "jgroups-udp" );
				socketBindingGroup.addChildResource( jgroupsUDP );
				Attribute multicast_address = client.readAttribute( jgroupsUDP, "multicast-address" );
				String newIPAddress = modifyIPAddress( multicast_address.getValue().asString(), offsetUnit );
				multicast_address.setValue( newIPAddress );
				client.writeAttribute( jgroupsUDP, multicast_address );
			}
			{
				Resource messagingGroup = new Resource( "socket-binding", "messaging-group" );
				socketBindingGroup.addChildResource( messagingGroup );
				Attribute multicast_address = client.readAttribute( messagingGroup, "multicast-address" );
				String newIPAddress = modifyIPAddress( multicast_address.getValue().asString(), offsetUnit );
				multicast_address.setValue( newIPAddress );
				client.writeAttribute( messagingGroup, multicast_address );
			}
		}
	}

	private void stopAllServers(List<Resource> serverGroups) throws IOException
	{
		for( Resource serverGroup : serverGroups )
		{
			client.operation( serverGroup, "stop-servers" );
			System.out.println( "Stopped servers in group " + serverGroup.getName() );
		}
	}

	private void removeServerConfigs(List<Resource> hosts) throws IOException
	{
		for( Resource host : hosts )
		{
			waitForServerShutdown( host );
			List<Resource> serverConfigs = client.getResourcesByType( host, "server-config" );
			for( Resource serverConfig : serverConfigs )
			{
				client.operation( serverConfig, "remove" );
				System.out.println( "Removed server " + serverConfig.getName() );
			}
		}
	}

	private void waitForServerShutdown(Resource host) throws IOException
	{
		while( isServerShutdown( host ) == false )
		{
			try
			{
				System.out.println( "Servers for host " + host.getName() + " not shut down yet, will wait and check again" );
				Thread.sleep( 1000 );
			}
			catch( InterruptedException e )
			{
				e.printStackTrace();
			}
		}
	}

	private boolean isServerShutdown(Resource host) throws IOException
	{
		try
		{
			client.getResourcesByType( host, "server" );
			//No exception means servers were found to not have been shut down
			return false;
		}
		catch( UnsuccessfulCommandException e )
		{
			//This means all servers have been shut down
			System.out.println( "Expected exception, which means that the server has been successfully shut down" );
			return true;
		}
	}

	private void removeServerGroups(List<Resource> serverGroups) throws IOException
	{
		for( Resource serverGroup : serverGroups )
		{
			client.operation( serverGroup, "remove" );
			System.out.println( "Removed server group " + serverGroup.getName() );
		}
	}

	private void createServerGroups() throws IOException
	{
		Resource clusterServerGroup1 = new Resource( "server-group", "cluster-server-group-1" );
		clusterServerGroup1.addAttribute( new Attribute( "profile", "full-ha-1" ) );
		clusterServerGroup1.addAttribute( new Attribute( "socket-binding-group", "full-ha-sockets" ) );
		client.create( clusterServerGroup1 );

		Resource clusterServerGroup2 = clusterServerGroup1;
		clusterServerGroup2.setName( "cluster-server-group-2" );
		clusterServerGroup2.setAttribute( "profile", "full-ha-2" );
		client.create( clusterServerGroup2 );

		Resource clusterServerGroup3 = clusterServerGroup1;
		clusterServerGroup3.setName( "cluster-server-group-3" );
		clusterServerGroup3.setAttribute( "profile", "full-ha-3" );
		client.create( clusterServerGroup3 );
	}

	private void createServerConfigs(List<Resource> hosts) throws IOException
	{
		for( int index = 0; index < hosts.size(); index++ )
		{
			Resource host = hosts.get( index );
			int serverGroup = ( index % 3 ) + 1;
			Resource serverConfig = new Resource( "server-config", host.getName() + "-" + domainName + "-server" );
			host.addChildResource( serverConfig );
			serverConfig.addAttribute( new Attribute( "auto-start", true ) );
			serverConfig.addAttribute( new Attribute( "group", "cluster-server-group-" + serverGroup ) );
			if( offsetUnit > 0 )
			{
				int portOffset = offsetUnit * 100;
				serverConfig.addAttribute( new Attribute( "socket-binding-port-offset", portOffset ) );
			}
			client.create( serverConfig );
			System.out.println( "Created server config " + serverConfig.getName() );
		}
	}

	private void startAllServers() throws IOException
	{
		List<Resource> serverGroups = client.getResourcesByType( null, "server-group" );
		for( Resource serverGroup : serverGroups )
		{
			client.operation( serverGroup, "start-servers" );
			System.out.println( "Started servers in group " + serverGroup.getName() );
		}
	}

	private String modifyIPAddress(String expression, int offsetUnit)
	{
		Matcher matcher = IP_ADDRESS_PATTERN.matcher( expression );
		if( matcher.matches() )
		{
			int octet4 = Integer.valueOf( matcher.group( 5 ) ) + 1;
			StringBuilder stringBuilder = new StringBuilder( matcher.group( 1 ) );
			stringBuilder.append( matcher.group( 2 ) );
			stringBuilder.append( "." );
			stringBuilder.append( matcher.group( 3 ) );
			stringBuilder.append( "." );
			stringBuilder.append( matcher.group( 4 ) );
			stringBuilder.append( "." );
			stringBuilder.append( octet4 );
			stringBuilder.append( matcher.group( 6 ) );
			return stringBuilder.toString();
		}
		else
		{
			return expression;
		}
	}
}