package org.jboss.refarch.eap6.cli;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Stack;

import org.jboss.as.cli.CommandContext;
import org.jboss.as.cli.CommandContextFactory;
import org.jboss.as.cli.CommandFormatException;
import org.jboss.as.cli.CommandLineException;
import org.jboss.as.controller.client.helpers.ClientConstants;
import org.jboss.dmr.ModelNode;

public class Client
{

	private CommandContext commandContext;

	public Client(String username, char[] password, String host, Integer port) throws CommandLineException
	{
		commandContext = CommandContextFactory.getInstance().newCommandContext( username, password );
		if( host != null )
		{
			commandContext.connectController( host, port );
		}
		else
		{
			commandContext.connectController();
		}
	}

	public void load(Resource resource) throws IOException
	{
		ModelNode readResourceRequest = getModelNode( resource );
		readResourceRequest.get( ClientConstants.OP ).set( "read-resource-description" );
		ModelNode readResourceResponse = execute( readResourceRequest );
		throwExceptionOnFailure( readResourceRequest, readResourceResponse, "Failed to read resource description for " + resource.getName() );
		ModelNode readResourceResult = readResourceResponse.get( ClientConstants.RESULT );
		ModelNode attributeModelNodes = readResourceResult.get( "attributes" );
		Collection<String> attributeNames = attributeModelNodes.keys();
		for( String attributeName : attributeNames )
		{
			if( !attributeName.equals( ClientConstants.NAME ) )
			{
				Attribute attribute = readAttribute( resource, attributeName, false );
				if( attribute.getValue().isDefined() )
				{
					resource.addAttribute( attribute );
				}
			}
		}
		ModelNode childTypeModelNodes = readResourceResult.get( "children" );
		Collection<String> childTypes = childTypeModelNodes.keys();
		for( String type : childTypes )
		{
			ModelNode readChildNamesRequest = getModelNode( resource );
			readChildNamesRequest.get( ClientConstants.OP ).set( ClientConstants.READ_CHILDREN_NAMES_OPERATION );
			readChildNamesRequest.get( ClientConstants.CHILD_TYPE ).set( type );
			ModelNode readChildNamesResponse = execute( readChildNamesRequest );
			throwExceptionOnFailure( readChildNamesRequest, readChildNamesResponse, "Failed to read children names for " + type );
			List<ModelNode> childNamesModelNodes = readChildNamesResponse.get( ClientConstants.RESULT ).asList();
			for( ModelNode childNameModelNode : childNamesModelNodes )
			{
				Resource childResource = new Resource( type, childNameModelNode.asString() );
				Collection<String> skipResourceTypes = new ArrayList<String>();
				skipResourceTypes.add( "access-log" );
				skipResourceTypes.add( "sso" );
				if( skipResourceTypes.contains( childResource.getType() ) )
				{
					//access-log cannot be read, skip this child resource
					continue;
				}
				childResource.setParent( resource );
				load( childResource );
				resource.addChildResource( childResource );
			}
		}
	}

	public void remove(Resource resource) throws IOException
	{
		Collection<String> skipResourceTypes = new ArrayList<String>();
		skipResourceTypes.add( "virtual-server" );
		skipResourceTypes.add( "stack" );
		if( !skipResourceTypes.contains( resource.getType() ) )
		{
			//stack itself is removed, not its transport and protocols
			for( Resource child : resource.getChildren() )
			{
				remove( child );
			}
		}
		ModelNode removeRequest = getModelNode( resource );
		removeRequest.get( ClientConstants.OP ).set( ClientConstants.REMOVE_OPERATION );
		ModelNode removeResponse = execute( removeRequest );
		throwExceptionOnFailure( removeRequest, removeResponse, "Failed to remove profile " + resource.getName() );
	}

	public void create(Resource resource) throws IOException
	{
		if( resource.getType().equals( "stack" ) )
		{
			createStack( resource );
			return;
		}
		ModelNode addResourceRequest = getModelNode( resource );
		addResourceRequest.get( ClientConstants.OP ).set( ClientConstants.ADD );
		for( Attribute attribute : resource.getAttributes() )
		{
			addResourceRequest.get( attribute.getName() ).set( attribute.getValue() );
		}
		if( isRequiredArgument( resource, ClientConstants.ADD, ClientConstants.NAME ) )
		{
			addResourceRequest.get( ClientConstants.NAME ).set( resource.getName() );
		}
		ModelNode addResourceResponse = execute( addResourceRequest );
		throwExceptionOnFailure( addResourceRequest, addResourceResponse, "Failed to create resource " + resource.getName() );
		Collection<String> skipResourceTypes = new ArrayList<String>();
		skipResourceTypes.add( "authorization" );
		skipResourceTypes.add( "authentication" );
		skipResourceTypes.add( "classic" );
		//Children are created as attributes
		if( !skipResourceTypes.contains( resource.getType() ) )
		{
			for( Resource child : resource.getChildren() )
			{
				create( child );
			}
		}
		if( resource.getType().equals( "data-source" ) && isEnabled( resource ) )
		{
			ModelNode enableDatasourceRequest = getModelNode( resource );
			enableDatasourceRequest.get( ClientConstants.OP ).set( "enable" );
			enableDatasourceRequest.get( "persistent" ).set( true );
			ModelNode enableDatasourceResponse = execute( enableDatasourceRequest );
			throwExceptionOnFailure( enableDatasourceRequest, enableDatasourceResponse, "Failed to enable datasource " + resource.getName() );
		}
	}

	public List<Resource> getResourcesByType(Resource parent, String type) throws IOException
	{
		ModelNode getResourcesRequest = getModelNode( parent );
		getResourcesRequest.get( ClientConstants.OP ).set( ClientConstants.READ_CHILDREN_NAMES_OPERATION );
		getResourcesRequest.get( ClientConstants.CHILD_TYPE ).set( type );
		ModelNode getResourcesResponse = execute( getResourcesRequest );
		throwExceptionOnFailure( getResourcesRequest, getResourcesResponse, "Failed to query resources of type " + type );
		List<ModelNode> resourceNames = getResourcesResponse.get( ClientConstants.RESULT ).asList();
		List<Resource> resources = new ArrayList<Resource>();
		for( ModelNode modelNode : resourceNames )
		{
			Resource resource = new Resource( type, modelNode.asString() );
			resource.setParent( parent );
			resources.add( resource );
		}
		return resources;
	}

	public ModelNode operation(Resource resource, String operation) throws IOException
	{
		ModelNode request = getModelNode( resource );
		request.get( ClientConstants.OP ).set( operation );
		ModelNode response = execute( request );
		throwExceptionOnFailure( request, response, "Failed to call " + operation + " on " + resource );
		return response;
	}

	public ModelNode deploy(String deployable) throws IOException, CommandFormatException
	{
		StringBuilder command = new StringBuilder( ClientConstants.DEPLOYMENT_DEPLOY_OPERATION );
		command.append( " " );
		command.append( deployable );
		command.append( " " );
		command.append( "--all-server-groups" );
		ModelNode request = commandContext.buildRequest( command.toString() );
		//avoid using execute() as it would print the deployable bytes and pollute the log
		System.out.println( "\n\n\n******************************************************************\n" );
		System.out.println( "Deploying " + deployable );
		ModelNode response = commandContext.getModelControllerClient().execute( request );
		throwExceptionOnFailure( request, response, "Failed to deploy " + deployable );
		return response;
	}

	public Attribute readAttribute(Resource resource, String attributeName) throws IOException
	{
		return readAttribute( resource, attributeName, true );
	}

	public Attribute readAttribute(Resource resource, String attributeName, boolean includeDefaults) throws IOException
	{
		ModelNode readAttributeRequest = getModelNode( resource );
		readAttributeRequest.get( ClientConstants.OP ).set( ClientConstants.READ_ATTRIBUTE_OPERATION );
		readAttributeRequest.get( ClientConstants.NAME ).set( attributeName );
		readAttributeRequest.get( "include-defaults" ).set( includeDefaults );
		ModelNode readAttributeResponse = execute( readAttributeRequest );
		throwExceptionOnFailure( readAttributeRequest, readAttributeResponse, "Failed to read attribute " + attributeName + " of " + resource );
		return new Attribute( attributeName, readAttributeResponse.get( ClientConstants.RESULT ) );
	}

	public ModelNode writeAttribute(Resource resource, Attribute attribute) throws IOException
	{
		ModelNode request = getModelNode( resource );
		request.get( ClientConstants.OP ).set( ClientConstants.WRITE_ATTRIBUTE_OPERATION );
		request.get( ClientConstants.NAME ).set( attribute.getName() );
		request.get( ClientConstants.VALUE ).set( attribute.getValue() );
		ModelNode response = execute( request );
		throwExceptionOnFailure( request, response, "Failed to write " + attribute + " to " + resource );
		return response;
	}

	public void disconnect()
	{
		commandContext.disconnectController();
	}

	private boolean isEnabled(Resource datasource)
	{
		for( Attribute attribute : datasource.getAttributes() )
		{
			if( attribute.getName().equals( "enabled" ) )
			{
				if( attribute.getValue().asBoolean() == true )
				{
					return true;
				}
			}
		}
		return false;
	}

	private void createStack(Resource stack) throws IOException
	{
		ModelNode createStackRequest = getModelNode( stack );
		createStackRequest.get( ClientConstants.OP ).set( ClientConstants.ADD );
		
		ModelNode transport = getModelNode( stack.getChild( "transport", "TRANSPORT" ).getAttributes() );

		List<ModelNode> protocols = new ArrayList<ModelNode>();
		//Use attributes to create protocols as children are out of order and the order is important
		for( ModelNode protocolModelNode : stack.getAttribute( "protocols" ).getValue().asList() )
		{
			Resource protocolChild = stack.getChild( "protocol", protocolModelNode.asString() );
			protocols.add( getModelNode( protocolChild.getAttributes() ) );
		}
		
		createStackRequest.get( "transport" ).set( transport );
		createStackRequest.get( "protocols" ).set( protocols );
		ModelNode createStackResponse = execute( createStackRequest );
		throwExceptionOnFailure( createStackRequest, createStackResponse, "Failed to create stack " + stack.getName() );
	}

	private ModelNode getModelNode(List<Attribute> attributes)
	{
		ModelNode modelNode = new ModelNode();
		for( Attribute attribute : attributes )
		{
			modelNode.get( attribute.getName() ).set( attribute.getValue() );
		}
		return modelNode;
	}

	private boolean isRequiredArgument(Resource resource, String operation, String requestProperty) throws IOException
	{
		ModelNode readOperationDescriptionRequest = getModelNode( resource );
		readOperationDescriptionRequest.get( ClientConstants.OP ).set( "read-operation-description" );
		readOperationDescriptionRequest.get( ClientConstants.NAME ).set( operation );
		ModelNode readOperationDescriptionResponse = execute( readOperationDescriptionRequest );
		throwExceptionOnFailure( readOperationDescriptionRequest, readOperationDescriptionResponse, "Failed to check reqired attributes for calling "
				+ operation + " on " + resource.getName() );
		ModelNode requestSpec = readOperationDescriptionResponse.get( ClientConstants.RESULT ).get( "request-properties" ).get( requestProperty );
		if( requestSpec.isDefined() )
		{
			return requestSpec.get( "required" ).asBoolean();
		}
		else
		{
			return false;
		}
	}

	private ModelNode getModelNode(Resource resource)
	{
		ModelNode modelNode = new ModelNode();
		if( resource != null )
		{
			Stack<Resource> addressStack = resource.getAddress();
			while( !addressStack.empty() )
			{
				Resource node = addressStack.pop();
				modelNode.get( ClientConstants.OP_ADDR ).add( node.getType(), node.getName() );
			}
		}
		return modelNode;
	}

	private ModelNode execute(ModelNode request) throws IOException
	{
		System.out.println( "\n\n\n******************************************************************\n" );
		System.out.println( request );
		return commandContext.getModelControllerClient().execute( request );
	}

	private void throwExceptionOnFailure(ModelNode request, ModelNode response, String message) throws UnsuccessfulCommandException
	{
		if( !response.get( ClientConstants.OUTCOME ).asString().equals( ClientConstants.SUCCESS ) )
		{
			System.out.println( "request: " + request );
			System.out.println( "response: " + response );
			throw new UnsuccessfulCommandException( message );
		}
	}
}