package org.jboss.refarch.bpms6;

import java.io.FileReader;
import java.io.IOException;
import java.util.List;
import java.util.Properties;

import org.jboss.as.cli.CommandLineException;
import org.jboss.dmr.ModelNode;
import org.jboss.refarch.eap6.cli.Attribute;
import org.jboss.refarch.eap6.cli.Client;
import org.jboss.refarch.eap6.cli.Resource;
import org.jboss.refarch.eap6.cli.UnsuccessfulCommandException;

public class BPMS
{

	private Client client;
	private String postgresDriverName;
	private String postgresUsername;
	private String postgresPassword;
	private String bpmsConnectionUrl;
	private String bpmsDS;
	private String bpmsNonJTA_DS;
	private String gitDir;
	private String indexDir;
	private String quartzProperties;
	private String helixClusterId;
	private String zookeeper;
	private String businessCentral;
	private String dashbuilder;

	public static void main(String[] args) throws CommandLineException, IOException
	{
		String propertyFile = "./configuration.properties";
		if( args.length == 1 )
		{
			propertyFile = args[0];
		}
		Properties props = new Properties();
		props.load( new FileReader( propertyFile ) );
		System.out.println( "properties loaded as: " + props );
		BPMS configuration = new BPMS( props );
		configuration.configure();
		System.out.println( "Done!" );
	}

	public BPMS(Properties properties) throws CommandLineException
	{
		String username = properties.getProperty( "username" );
		String password = properties.getProperty( "password" );
		String domainController = properties.getProperty( "domainController" );
		client = new Client( username, password.toCharArray(), domainController, 9999 );
		postgresDriverName = properties.getProperty( "postgresDriverName" );
		postgresUsername = properties.getProperty( "postgresUsername" );
		postgresPassword = properties.getProperty( "postgresPassword" );

		bpmsConnectionUrl = properties.getProperty( "bpmsConnectionUrl" );
		bpmsDS = properties.getProperty( "bpmsDS" );
		bpmsNonJTA_DS = properties.getProperty( "bpmsNonJTA_DS" );
		gitDir = properties.getProperty( "gitDir" );
		indexDir = properties.getProperty( "indexDir" );
		quartzProperties = properties.getProperty( "quartzProperties" );
		helixClusterId = properties.getProperty( "helixClusterId" );
		zookeeper = properties.getProperty( "zookeeper" );
		businessCentral = properties.getProperty( "businessCentral" );
		dashbuilder = properties.getProperty( "dashbuilder" );
	}

	private void configure() throws CommandLineException, IOException
	{
		List<Resource> hosts = client.getResourcesByType( null, "host" );
		setSystemProperties( hosts );
		setJVM( hosts );

		List<Resource> profiles = client.getResourcesByType( null, "profile" );
		setClusteredSSO( profiles );
		setupDataSources( profiles );

		stopAllServers();
		for( Resource host : hosts )
		{
			waitForServerShutdown( host );
		}
		startAllServers();

		client.deploy( businessCentral );
		client.deploy( dashbuilder );

		client.disconnect();
		System.out.println( "Disconnected" );
	}

	private void setSystemProperties(List<Resource> hosts) throws IOException
	{
		String[][] newProperties = new String[2][];
		newProperties[0] = new String[] {"org.uberfire.nio.git.dir", "org.uberfire.metadata.index.dir", "org.quartz.properties", "jboss.node.name", "org.uberfire.cluster.id", "org.uberfire.cluster.zk", "org.uberfire.cluster.local.id", "org.uberfire.cluster.vfs.lock", "org.uberfire.nio.git.deamon.port", "org.uberfire.cluster.autostart", "org.uberfire.nio.git.ssh.port"};
		newProperties[1] = new String[] {gitDir, indexDir, quartzProperties, null, helixClusterId, zookeeper, null, "vfs-repo", "9418", "false", "8003"};

		for( Resource host : hosts )
		{
			newProperties[1][3] = host.getName();

			String bindAddress;
			Resource server = client.getResourcesByType( host, "server" ).get( 0 );
			Resource platformMBean = new Resource( "core-service", "platform-mbean" );
			platformMBean.setParent( server );
			Resource runtime = new Resource( "type", "runtime" );
			runtime.setParent( platformMBean );
			client.load( runtime );
			ModelNode bindAddressNode = runtime.getAttribute( "system-properties" ).getValue().get( "jboss.bind.address" );
			if( bindAddressNode.isDefined() )
			{
				bindAddress = bindAddressNode.asString();
			}
			else
			{
				bindAddress = null;
			}

			List<Resource> serverConfigs = client.getResourcesByType( host, "server-config" );
			for( Resource serverConfig : serverConfigs )
			{
				Attribute portOffsetAttr = client.readAttribute( serverConfig, "socket-binding-port-offset" );
				int port = 8080 + portOffsetAttr.getValue().asInt();
				newProperties[1][6] = newProperties[1][3] + "_" + port;

				Resource systemProperty = new Resource( "system-property", null );
				systemProperty.setParent( serverConfig );
				systemProperty.addAttribute( new Attribute( "boot-time", false ) );
				for( int index = 0; index < newProperties[0].length; index++ )
				{
					systemProperty.setName( newProperties[0][index] );
					systemProperty.setAttribute( "value", newProperties[1][index] );
					client.create( systemProperty );
				}
				if( bindAddress != null )
				{
					Resource bindAddressProperty = new Resource( "system-property", "org.uberfire.nio.git.daemon.host" );
					bindAddressProperty.setParent( serverConfig );
					bindAddressProperty.setAttribute( "value", bindAddress );
					client.create( bindAddressProperty );
					bindAddressProperty.setName( "org.uberfire.nio.git.ssh.host" );
					client.create( bindAddressProperty );
				}
			}
		}
	}

	private void setJVM(List<Resource> hosts) throws IOException
	{
		for( Resource host : hosts )
		{
			List<Resource> serverConfigs = client.getResourcesByType( host, "server-config" );
			for( Resource serverConfig : serverConfigs )
			{
				Resource jvm = new Resource( "jvm", "serverJVM" );
				jvm.setParent( serverConfig );
				jvm.setAttribute( "max-permgen-size", "512m" );
				client.create( jvm );
			}
		}
	}

	private void setClusteredSSO(List<Resource> profiles) throws IOException
	{
		for( Resource profile : profiles )
		{
			Resource web = new Resource( "subsystem", "web" );
			web.setParent( profile );
			List<Resource> virtualServers = client.getResourcesByType( web, "virtual-server" );
			for( Resource virtualServer : virtualServers )
			{
				List<Resource> ssoConfigs = client.getResourcesByType( virtualServer, "sso" );
				for( Resource sso : ssoConfigs )
				{
					client.remove( sso );
				}
				Resource sso = new Resource( "sso", "configuration" );
				sso.setParent( virtualServer );
				sso.setAttribute( "cache-container", "web" );
				sso.setAttribute( "cache-name", "sso" );
				sso.setAttribute( "domain-name", "bpms" );
				sso.setAttribute( "reauthenticate", false );
				client.create( sso );
			}
		}
	}

	private void setupDataSources(List<Resource> profiles) throws IOException
	{
		Resource dataSources = new Resource( "subsystem", "datasources" );

		Resource bpmsJTA = new Resource( "data-source", bpmsDS );
		bpmsJTA.setParent( dataSources );
		bpmsJTA.setAttribute( "enabled", true );
		bpmsJTA.setAttribute( "jta", true );
		bpmsJTA.setAttribute( "jndi-name", "java:jboss/datasources/" + bpmsDS );
		bpmsJTA.setAttribute( "connection-url", bpmsConnectionUrl );
		bpmsJTA.setAttribute( "driver-class", "org.postgresql.xa.PGXADataSource" );
		bpmsJTA.setAttribute( "driver-name", postgresDriverName );
		bpmsJTA.setAttribute( "user-name", postgresUsername );
		bpmsJTA.setAttribute( "password", postgresPassword );
		bpmsJTA.setAttribute( "use-java-context", true );
		bpmsJTA.setAttribute( "use-ccm", true );

		Resource bpmsNonJTA = new Resource( "data-source", bpmsNonJTA_DS );
		bpmsNonJTA.setParent( dataSources );
		bpmsNonJTA.setAttribute( "enabled", true );
		bpmsNonJTA.setAttribute( "jta", false );
		bpmsNonJTA.setAttribute( "jndi-name", "java:jboss/datasources/" + bpmsNonJTA_DS );
		bpmsNonJTA.setAttribute( "connection-url", bpmsConnectionUrl );
		bpmsNonJTA.setAttribute( "driver-class", "org.postgresql.Driver" );
		bpmsNonJTA.setAttribute( "driver-name", postgresDriverName );
		bpmsNonJTA.setAttribute( "user-name", postgresUsername );
		bpmsNonJTA.setAttribute( "password", postgresPassword );
		bpmsNonJTA.setAttribute( "use-java-context", true );
		bpmsNonJTA.setAttribute( "use-ccm", true );

		for( Resource profile : profiles )
		{
			dataSources.setParent( profile );
			client.create( bpmsJTA );
			client.create( bpmsNonJTA );
		}
	}

	private void waitForServerShutdown(Resource host) throws IOException
	{
		while( isServerShutdown( host ) == false )
		{
			try
			{
				System.out.println( "Servers for host " + host.getName() + " not shut down yet, will wait and check again" );
				Thread.sleep( 1000 );
			}
			catch( InterruptedException e )
			{
				e.printStackTrace();
			}
		}
	}

	private boolean isServerShutdown(Resource host) throws IOException
	{
		try
		{
			client.getResourcesByType( host, "server" );
			//No exception means servers were found to not have been shut down
			return false;
		}
		catch( UnsuccessfulCommandException e )
		{
			//This means all servers have been shut down
			System.out.println( "Expected exception, which means that the server has been successfully shut down" );
			return true;
		}
	}

	private void stopAllServers() throws IOException
	{
		List<Resource> serverGroups = client.getResourcesByType( null, "server-group" );
		for( Resource serverGroup : serverGroups )
		{
			client.operation( serverGroup, "stop-servers" );
			System.out.println( "Stopped servers in group " + serverGroup.getName() );
		}
	}

	private void startAllServers() throws IOException
	{
		List<Resource> serverGroups = client.getResourcesByType( null, "server-group" );
		for( Resource serverGroup : serverGroups )
		{
			client.operation( serverGroup, "start-servers" );
			System.out.println( "Started servers in group " + serverGroup.getName() );
		}
	}
}