#! /bin/sh.


# -------------------------------------
# SPDX-License-Identifier: MIT-0
# -------------------------------------


CONFIG_FILE="$HOME/properties.json"		#[1] Absolute File Path to properties.json
server='access.redhat.com'
tokenApi='/hydra/rest/v2/sftp/token'
client_id='hydra-sftp'
device_auth_url='https://sso.redhat.com/auth/realms/redhat-external/protocol/openid-connect/auth/device'
sso_token_url='https://sso.redhat.com/auth/realms/redhat-external/protocol/openid-connect/token'
log_sso_token=0

if type "yum" > /dev/null 2>&1; then
  if ! type "jq" > /dev/null 2>&1; then
    yum install -q -y jq > /dev/null 2>&1
  fi
  if ! type "expect" > /dev/null 2>&1; then
    yum install -q -y expect > /dev/null 2>&1
  fi
elif type "brew" > /dev/null 2>&1; then
  if ! type "jq" > /dev/null 2>&1; then
    brew install -q -y jq > /dev/null 2>&1
  fi
  if ! type "expect" > /dev/null 2>&1; then
    brew install -q -y expect > /dev/null 2>&1
  fi
fi

flag=1
if ! type "jq" > /dev/null 2>&1; then
  printf "\033[1;31mPlease ensure 'jq' is installed to use this script.\033[0m\n"
  flag=0
fi
if ! type "expect" > /dev/null 2>&1; then
  printf "\033[1;31mPlease ensure 'expect' is installed to use this script.\033[0m\n"
  flag=0
fi
if [[ $flag != 1 ]]; then
  exit 1
fi

############################################################
# Help                                                     #
############################################################
function Help() {
  # Display Help
  echo
  echo "DESCRIPTION:"
  echo "This wrapper script eliminates the multiple commands needed to be typed manually by the user to upload a file to SFTP. It takes the file path for the file to be uploaded and then performs all the steps from authentication to uploading the file to SFTP. The user should note that the script makes use of v2 API for SFTP token generation."
  echo
  echo "SYNTAX: sh uploadfiletosftp.sh [--path file_path] [-C case_number] [--proxy proxy_type] [-x proxy_user] [-y proxy_password] [-h proxy_host] [-o proxy_port] [-c] [-b] [--help]"
  echo
  echo "OPTIONS:"
  echo "      --path            The --path option is a mandatory option, whose option argument is used to specify the complete file path of the file to be uploaded."
  echo "-C  | --case            If the -C option is present, the corresponding option argument is used to specify the case number to which the file is to be attached."
  echo "-x  | --proxy_user      If the -x option is present, the value for the proxy_user in the properties.json file is set as the corresponding option argument."
  echo "-h  | --proxy_host      If the -h option is present, the value for the proxy_host in the properties.json file is set as the corresponding option argument."
  echo "-o  | --proxy_port      If the -o option is present, the value for the proxy_port in the properties.json file is set as the corresponding option argument."
  echo "-y  | --proxy_password  If the -y option is present, the corresponding option argument is used to specify the proxy_password of proxy_user. If your password contains any special character, you must enclose the 'password' in single quotes."
  echo "-c  | --config_all      If the -c option is present, it will trigger input prompts for configuring each property in properties.json file, enabling the user to set or change value for any property."
  echo "      --proxy           If the --proxy option is present, the corresponding option argument is used to specify the type of proxy to be used to connect to connect to SFTP. For authenticated proxy, provide the option argument as 'auth' and 'unauth' for an unauthenticated proxy connection. Any other value shall default to connect to SFTP without any proxy."
  echo "-b  | --sso_token       If the -b option is present, the script will print the bearer token corresponding to the customer portal account of the user and exit."
  echo "      --help            Print this help and exit."
  echo
}

############################################################
############################################################

function configAll() {
  get_config
  printf "\033[1;42m\nConfiguring Properties...\nPRESS 'ENTER' IF YOU DO NOT WISH TO ALTER THE VALUE\033[0m\n\n"
  read -rp "Proxy[$proxy]: " proxy
  read -rp "Proxy_user[$proxy_user]: " proxy_user
  read -rsp "Proxy_password: " proxy_password
  echo
  read -rp "Proxy_host[$proxy_host]: " proxy_host
  read -rp "Proxy_port[$proxy_port]: " proxy_port
  printf "\n\e[1;32mConfiguration saved\e[0m\n\n"
}

function readConfig() {
  proxy=$( < "$CONFIG_FILE" jq -r '.proxy')
  proxy_user=$( < "$CONFIG_FILE" jq -r '.proxy_user')
  proxy_host=$( < "$CONFIG_FILE" jq -r '.proxy_host')
  proxy_port=$( < "$CONFIG_FILE" jq -r '.proxy_port')
}

function setConfig() {
  local tmp=" "
  if [[ -n "$proxy" ]] ; then jq --arg a "$1" '.proxy=$a' "$CONFIG_FILE" > "$tmp" && mv "$tmp" "$CONFIG_FILE"
  fi
  shift
  if [[ -n "$proxy_user" ]] ; then jq --arg a "$1" '.proxy_user=$a' "$CONFIG_FILE" > "$tmp" && mv "$tmp" "$CONFIG_FILE"
  fi
  shift
  if [[ -n "$proxy_host" ]] ; then jq --arg a "$1" '.proxy_host=$a' "$CONFIG_FILE" > "$tmp" && mv "$tmp" "$CONFIG_FILE"
  fi
  shift
  if [[ -n "$proxy_port" ]] ; then
    if ! [[ $1 =~ ^[0-9]+$ ]] ; then
      printf "\033[1;31mInvalid Proxy_port: %s\033[0m\n" "$1">&2;
      exit 1
    else
      jq ".proxy_port=$1" "$CONFIG_FILE" > "$tmp" && mv "$tmp" "$CONFIG_FILE"
    fi
  fi
  shift
}

function ensureConfigFileExists() {
  if [ ! -e "$1" ] ; then
    if [ -e "$1.example" ]; then
      cp "$1.example" "$1";
    else
      touch "$CONFIG_FILE"
      printf '{\n"proxy" : "none",\n"proxy_user" : "",\n"proxy_host" : "",\n"proxy_port" : 22\n}' > "$CONFIG_FILE"
    fi
  fi
  chmod go-rwx,u+rwx "$1"
}

updateConfig() {
  local key="$1"
  local value="$2"
  local tmp=$(mktemp)

  jq --arg key "$key" --arg value "$value" '.[$key]=$value' "$CONFIG_FILE" > "$tmp" && mv "$tmp" "$CONFIG_FILE"
}

getConfig() {
  local key="$1"
  local value=$(jq -r ".$key" "$CONFIG_FILE")
  echo "$value"
}

deviceAuthFlow() {
  device_auth_response="$(curl -sk --request POST "${device_auth_url}" --header 'Content-Type: application/x-www-form-urlencoded' --data-urlencode "client_id=${client_id}")"
  start_time=$(date +%s)
  end_time=$((start_time + 540))
  verification_uri_complete=$(echo "$device_auth_response" | jq -r '.verification_uri_complete')
  device_code=$(echo "$device_auth_response" | jq -r '.device_code')

  fallback() {
      printf "Please visit the following url to login to your Red Hat Customer Portal account -\n%s\n" "${verification_uri_complete}" >&2
  }
  if command -v xdg-open > /dev/null 2>&1; then
      if ! xdg-open "$verification_uri_complete" 2>/dev/null >/dev/null; then
          fallback
      fi
  elif command -v open > /dev/null 2>&1; then
      if ! open "$verification_uri_complete" 2>/dev/null >/dev/null; then
          fallback
      fi
  else
      fallback
  fi

  while [ $(date +%s) -lt $end_time ]; do

    http_response=$(curl -sk --request POST "${sso_token_url}" --header 'Content-Type: application/x-www-form-urlencoded' --data-urlencode "client_id=${client_id}" --data-urlencode 'grant_type=urn:ietf:params:oauth:grant-type:device_code' --data-urlencode "device_code=${device_code}" -sw '\n%{http_code}' -o -)
    http_status_code=$(echo "$http_response" | tail -n1)
    sso_token_response=$(echo "$http_response" | head -n1)

    if [ "$http_status_code" -eq 200 ]; then
      break
    fi
    
    if [ "$http_status_code" -eq 400 ]; then
      if echo "$sso_token_response" | jq -e '.error' >/dev/null 2>&1; then
        error=$(echo "$sso_token_response" | jq -r '.error')
        if [ "$error" = "access_denied" ]; then
          message="Authorization request has been denied."
          printf "\033[1;31m%s\033[0m\n" "$message" >&2;
          exit 1
        fi
      fi
    fi

    # Wait for 5 seconds before trying again
    sleep 5
  done

  if [ "$http_status_code" -ne 200 ]; then
    message="Verification timed out. Please try again!"
    printf "\033[1;31m%s\033[0m\n" "$message" >&2;
    exit 1
  fi
  echo "${sso_token_response}"
}

ensureConfigFileExists "${CONFIG_FILE}"

while true; do
  case "$1" in
    --path ) filepath="$2"; shift 2;;
    --proxy_user | -x ) proxy_user="$2"; shift 2 ;;
    --proxy_host | -h ) proxy_host="$2"; shift 2 ;;
    --proxy_port | -o ) proxy_port="$2"; shift 2 ;;
    --proxy_password | -y) proxy_password="$2"; shift 2;;
    --case | -C) case_number="$2"; shift 2;;
    --config_all | -c ) configAll; shift;;
    --proxy ) proxy="$2"; shift 2;;
    --sso_token | -b) log_sso_token=1; shift;;
    --help ) Help; exit;;
    -- ) shift; break ;;
    * ) if [ ! "$1" ] ; then break
        else printf "\033[1;31mError: Invalid option %s\033[0m\nTry 'sh uploadFileToSFTP.sh --help' for more information.\n" "$1"; exit 1
        fi ;;
  esac
done

setConfig "$proxy" "$proxy_user" "$proxy_host" "$proxy_port"
readConfig

if [ "$case_number" ] && [ "$filepath" ]; then
  re='^[0-9]+$'
  if  [[ ! "$case_number" =~ $re ]] ; then
      message="Invalid case number!"
      printf "\033[1;31m%s\033[0m\n" "$message" >&2; exit 1
  fi
  if [ -e "$filepath" ] ; then
    file="/tmp/${case_number}_$(basename -- $filepath)"
    cp "$filepath" "$file"
    else
      message="File ${filepath} does not exist!"
      printf "\033[1;31m%s\033[0m\n" "$message" >&2; exit 1
  fi
fi

if [ "$proxy" == "auth" ]; then
  if [ ! "$proxy_user" ]; then
    read -rp "Proxy username: " proxy_user;
  fi
  if [ ! "$proxy_password" ]; then
    read -rsp "Password for proxy user: " proxy_password;
    echo
  fi
fi

if [ -z "$sso_token_response" ]; then
    sso_token_response=$(deviceAuthFlow)
fi

if [ $log_sso_token -eq 1 ]; then
    echo "${sso_token_response}"
    exit 1;
fi

if [ ! "$filepath" ]; then
  read -rep "File path: " filepath;
fi
file=$filepath

access_token=$(echo "$sso_token_response" | jq -r '.access_token')

if [ -n "${access_token}" ]; then
  echo "Authentication successful!"
  echo "Fetching SFTP token..."
  sftp_token_response="$(curl -sk --request POST https://${server}${tokenApi} --header "Authorization: Bearer ${access_token}")"
  sftp_token=$(echo "$sftp_token_response" | jq -r '.token')
  username=$(echo "$sftp_token_response" | jq -r '.username')
  printf "Connecting to %s@sftp.%s..." "$username" "$server"

  if [[ "$proxy" == "unauth" ]]; then
    con_string="sftp -oStrictHostKeyChecking=no \"-oProxyCommand nc --proxy ${proxy_host}:${proxy_port} --proxy-type http %h %p\" ${username}@sftp.${server}"
  elif [[ "$proxy" == "auth" ]]; then
    con_string="sftp -oStrictHostKeyChecking=no \"-oProxyCommand nc --proxy ${proxy_host}:${proxy_port} --proxy-auth ${proxy_user}:${proxy_password} --proxy-type http %h %p\" ${username}@sftp.${server}"
  else con_string="sftp -oStrictHostKeyChecking=no ${username}@sftp.${server}"
  fi

  expect -c "
  log_user 0
  spawn -noecho ${con_string}
  expect -re \"Connection closed by remote host\" {
    send_user \"\nUnable to authenticate via proxy\n\"
    exit
  }
  expect -re \"spawn id exp\" {
    send_user \"\nUnexpected response while connecting to SFTP server\n\"
    exit
  }
  expect -re \"password: \" {
    log_user 1
    send \"${sftp_token}\r\"
    log_user 0
    expect -re \"${server}\" {
      send_user \"\nConnected to ${username}@sftp.${server}\n\"
    }
  }
  expect -re \"Connection reset by peer\" {
    send_user \"\nUnexpected response while connecting to SFTP server\n\"
    exit
  }
  expect -re \"spawn id exp\" {
    send_user \"\nUnexpected response while connecting to SFTP server\n\"
    exit
  }
  expect -re \"sftp> \" {
    set timeout -1
    send \"put ${file}\r\"
  }
  expect -re \"${file}\" {
    log_user 1
  }
  expect -re \"Uploading ${file}\" {
    log_user 0
  }
  set timeout 2
  expect -re \"Permission denied\" {
    send_error \"File name already exists! Please rename the file and try again.\n\"
    exit
  }
  log_user 1
  set timeout -1
  expect -re \"sftp> \" {
    send_user \"exit\r\"
  }"

  if [[ "$file" != "$filepath" ]]; then
    rm "$file"
  fi
fi