/*
 * This is an example script to demonstrate how to update the configuration 
 * property of a resource's connection settings.
 * 
 * updateResourceDomainHost.js [<resource id> <domain host value>]
 * 
 * If no argument are provided, the script will output a list of resources of type 
 * JBossAS7 Host Controller. 
 * 
 * The expected arguments are the resource id of the resource to be updated and the 
 * value to assign to its domain host connection setting property.
 * 
 * To the extent possible under law, Red Hat, Inc. has dedicated all copyright to 
 * this software to the public domain worldwide, pursuant to the CC0 Public Domain 
 * Dedication. This software is distributed without any warranty.  
 * 
 * See <http://creativecommons.org/publicdomain/zero/1.0/>.
 *
 */

/**
 * Print a list of resources of type JBossAS7 Host Controller.
 * 
 * The list provides the resource id, name, lineage, key, and its existing 
 * domain host value. The list may assist with determining the resource id
 * of resources for which the domain host value should be updated.
 */
function printResources() {
    criteria = new ResourceCriteria();
    criteria.addFilterResourceTypeName('JBossAS7 Host Controller');
    criteria.setStrict(true);
    var resources = ResourceManager.findResourcesByCriteria(criteria);

    for (var i = 0; i < resources.size(); i++) {
        var resource = resources.get(i);
        println("  Resource ID: " + resource.getId());
        println("Resource Name: " + resource.getName());
        println("     Ancestry: " + resource.getAncestry());
        println(" Resource Key: " + resource.getResourceKey());
        println("  Domain Host: " + ConfigurationManager.getPluginConfiguration(resource.getId()).getSimpleValue("domainHost"));
        println("");
    }
}

/**
 * Update a resource's plug-in configuration domain host value.
 * 
 * The plug-in configuration domainHost value is updated for resourceId 
 * with the value specified by newDomainHost.
 * 
 * This method does not validate that the property exists or should or can
 * be updated.
 */
function updateResourceDomainHost(resourceId, newDomainHost) {
    var config = ConfigurationManager.getPluginConfiguration(resourceId);
    config.setSimpleValue("domainHost",newDomainHost);
    ConfigurationManager.updatePluginConfiguration(resourceId, config);
}


if (args.length < 1) { // if we have no arguments, print the list
    printResources();
} else if (args.length === 2) { // if we have 2 arguments then assume we have id and new value
    updateResourceDomainHost(args[0], args[1]); 
} else { // our simple usage message
    println("script [<resource id> <new domain host value>]");
}
