/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.redhat.refarch.hacep.rest.commands;

import com.redhat.refarch.hacep.dto.NodeType;
import com.redhat.refarch.hacep.dto.SessionDataObjectInformation;
import com.redhat.refarch.hacep.rest.RestInterface;
import com.redhat.refarch.hacep.util.JDGUtility;
import it.redhat.hacep.configuration.HACEPApplication;
import it.redhat.hacep.configuration.annotations.HACEPSessionCache;
import it.redhat.hacep.model.Key;
import org.infinispan.Cache;
import org.infinispan.manager.DefaultCacheManager;
import org.infinispan.remoting.transport.Address;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.inject.Inject;
import java.util.*;

public class InfoInterfaceRequest implements InterfaceRequest {

    private static final String COMMAND_NAME = "info";

    private final HACEPApplication application;

    @Inject
    private JDGUtility jdgUtility;

    @Inject
    @HACEPSessionCache
    private Cache<Key, Object> sessionCache;

    @Inject
    public InfoInterfaceRequest(HACEPApplication application) {
        this.application = application;
    }

    @Override
    public String command() {
        return COMMAND_NAME;
    }

    @Override
    public boolean execute(RestInterface console, Iterator<String> args) throws IllegalArgumentException {
        try {
            DefaultCacheManager cacheManager = application.getCacheManager();
            console.println(generalInfo());
            for (String cacheName : cacheManager.getCacheNames()) {
                console.println("\n");
                console.println(buildInfo(cacheManager.getCache(cacheName)));
            }

            console.println("\n\nSession cache objects:");
            Map<Address, List<SessionDataObjectInformation>> sessions = new HashMap<>();
            for (Map.Entry<Key, List<Address>> entry : jdgUtility.getKeysAddresses(sessionCache).entrySet()) {
                List<Address> addresses = entry.getValue() != null ? entry.getValue() : Collections.emptyList();
                for (int i = 0; i < addresses.size(); i++) {
                    boolean isPrimary = (i == 0);
                    sessions.compute(addresses.get(i), (a, l) -> {
                        if (l == null) {
                            l = new ArrayList<>();
                        }
                        SessionDataObjectInformation object = new SessionDataObjectInformation(entry.getKey().toString(), isPrimary ? NodeType.PRIMARY : NodeType.REPLICA);
                        l.add(object);
                        return l;
                    });
                }
            }

            console.print(sessions);

        } catch (NoSuchElementException e) {
            console.println(generalInfo());
        }
        return true;

    }

    private String generalInfo() {
        DefaultCacheManager cacheManager = application.getCacheManager();
        StringBuilder info = new StringBuilder();
        info.append("Cache Manager Status: ").append(cacheManager.getStatus()).append("\n");
        info.append("Cache Manager Address: ").append(cacheManager.getAddress()).append("\n");
        info.append("Coordinator address: ").append(cacheManager.getCoordinator()).append("\n");
        info.append("Is Coordinator: ").append(cacheManager.isCoordinator()).append("\n");
        info.append("Cluster Name: ").append(cacheManager.getClusterName()).append("\n");
        info.append("Cluster Size: ").append(cacheManager.getClusterSize()).append("\n");
        info.append("Member list: ").append(cacheManager.getMembers()).append("\n");
        info.append("Caches: ").append(cacheManager.getCacheNames()).append("\n");
        return info.toString();
    }

    private String buildInfo(Cache<Key, Object> cache) {

        StringBuilder info = new StringBuilder();

        info.append("Cache: ").append(cache).append("\n");
        info.append("Cache Mode: ").append(cache.getCacheConfiguration().clustering().cacheModeString()).append("\n");
        info.append("Cache Size: ").append(cache.size()).append("\n");
        info.append("Cache Status: ").append(cache.getStatus()).append("\n");
        info.append("Number of Owners: ").append(cache.getAdvancedCache().getDistributionManager().getConsistentHash().getNumOwners()).append("\n");
        info.append("Number of Segments: ").append(cache.getAdvancedCache().getDistributionManager().getConsistentHash().getNumSegments()).append("\n");

        return info.toString();
    }

    @Override
    public void usage(RestInterface console) {
        console.println(COMMAND_NAME);
        console.println("\t\tGeneral information on caches.");
    }
}
