/*
 * JBoss, Home of Professional Open Source
 * Copyright 2012, Red Hat, Inc. and/or its affiliates, and individual
 * contributors by the @authors tag. See the copyright.txt in the
 * distribution for a full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.as.server.deployment.scanner;

import java.util.Collection;
import java.util.EnumSet;
import java.util.logging.Logger;

import javax.annotation.PostConstruct;
import javax.ejb.Singleton;
import javax.ejb.Startup;

import org.jboss.as.clustering.singleton.SingletonService;
import org.jboss.as.controller.ModelController;
import org.jboss.as.server.CurrentServiceContainer;
import org.jboss.as.server.ServerEnvironment;
import org.jboss.as.server.ServerEnvironmentService;
import org.jboss.as.server.Services;
import org.jboss.msc.service.AbstractServiceListener;
import org.jboss.msc.service.ServiceController;
import org.jboss.msc.service.ServiceController.Transition;
import org.jboss.msc.service.ServiceListener;


/**
 * A Singleton EJB to create the SingletonService during startup.
 *
 * @author <a href="mailto:wfink@redhat.com">Wolf-Dieter Fink</a>
 */
@Singleton
@Startup
public class HAStartupSingleton {
	private static final Logger LOGGER = Logger.getLogger(HAStartupSingleton.class.getName());

	/**
	 * Create the Service and wait until it is started.<br/>
	 * Will log a message if the service will not start in 10sec.
	 */
	@PostConstruct
	protected void startup() {
		LOGGER.info(HAStartupSingleton.class.getSimpleName()+" will be initialized!");

		HADeployService service = new HADeployService();
		SingletonService<String> singleton = new SingletonService<String>(service, HADeployService.SINGLETON_SERVICE_NAME);
		ServiceController<String> controller = singleton.build(CurrentServiceContainer.getServiceContainer())
				.addDependency(ServerEnvironmentService.SERVICE_NAME, ServerEnvironment.class, service.getEnvInjector())
				.addDependency(Services.JBOSS_SERVER_CONTROLLER, ModelController.class, service.controllerValue)
				.install();

		controller.setMode(ServiceController.Mode.ACTIVE);
		try {
			wait(controller, EnumSet.of(ServiceController.State.DOWN, ServiceController.State.STARTING), ServiceController.State.UP);
			LOGGER.info(HAStartupSingleton.class.getSimpleName()+" has started the Service");
		} catch (IllegalStateException e) {
			LOGGER.warning(HAStartupSingleton.class.getSimpleName()+" "+HADeployService.SINGLETON_SERVICE_NAME+" not started, are you sure to start in a cluster (HA) environment?");
		}
	}


	private static <T> void wait(ServiceController<T> controller, Collection<ServiceController.State> expectedStates, ServiceController.State targetState) {
		if (controller.getState() != targetState) {
			ServiceListener<T> listener = new NotifyingServiceListener<T>();
			controller.addListener(listener);
			try {
				synchronized (controller) {
					int maxRetry = 2;
					while (expectedStates.contains(controller.getState()) && maxRetry > 0) {
						LOGGER.info("Service controller state is "+controller.getState()+", waiting for transition to "+targetState);
						controller.wait(5000);
						maxRetry--;
					}
				}
			} catch (InterruptedException e) {
				LOGGER.warning("Wait on startup is interrupted!");
				Thread.currentThread().interrupt();
			}
			controller.removeListener(listener);
			ServiceController.State state = controller.getState();
			LOGGER.info("Service controller state is now "+state);
			if (state != targetState) {
				throw new IllegalStateException(String.format("Failed to wait for state to transition to %s.  Current state is %s", targetState, state), controller.getStartException());
			}
		}
	}

	private static class NotifyingServiceListener<T> extends AbstractServiceListener<T> {
		@Override
		public void transition(ServiceController<? extends T> controller, Transition transition) {
			synchronized (controller) {
				controller.notify();
			}
		}
	}
}
