package org.jboss.as.server.deployment.scanner;

import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.ARCHIVE;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.COMPOSITE;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.CONTENT;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.DEPLOYMENT;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.FAILED;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.FAILURE_DESCRIPTION;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.OUTCOME;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.PATH;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.PERSISTENT;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.STEPS;

import java.io.File;
import java.security.AccessController;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jboss.as.controller.ModelController;
import org.jboss.as.controller.client.ModelControllerClient;
import org.jboss.as.controller.operations.common.Util;
import org.jboss.as.server.ServerEnvironment;
import org.jboss.as.server.deployment.DeploymentAddHandler;
import org.jboss.as.server.deployment.DeploymentDeployHandler;
import org.jboss.as.server.deployment.DeploymentRemoveHandler;
import org.jboss.as.server.deployment.DeploymentUndeployHandler;
import org.jboss.dmr.ModelNode;
import org.jboss.msc.service.Service;
import org.jboss.msc.service.ServiceName;
import org.jboss.msc.service.StartContext;
import org.jboss.msc.service.StartException;
import org.jboss.msc.service.StopContext;
import org.jboss.msc.value.InjectedValue;
import org.jboss.threads.JBossThreadFactory;

/**
 * This service deploy all EAR, JAR, WAR files from the folder standalone/ha-deployments.
 * There is no support for exploded deployments and scanning implemneted.
 * 
 * It can be used as a workaround until the full feature is implemented in AS7/EAP6.
 * 
 * @author <a href="mailto:wfink@redhat.com">Wolf-Dieter Fink</a>
 */
public class HADeployService implements Service<String> {
	private static final Logger LOGGER = Logger.getLogger(HADeployService.class.getName());
	static final ServiceName SINGLETON_SERVICE_NAME = ServiceName.JBOSS.append("ha", "singleton", "deployer");
	private final AtomicBoolean started = new AtomicBoolean(false);
	protected final InjectedValue<ServerEnvironment> env = new InjectedValue<ServerEnvironment>();
	final InjectedValue<ModelController> controllerValue = new InjectedValue<ModelController>();

	private final ArrayList<String> deployed = new ArrayList<String>();

	public InjectedValue<ServerEnvironment> getEnvInjector() {
		return this.env;
	}

	@Override
	public String getValue() throws IllegalStateException, IllegalArgumentException {
		return null;
	}

	@Override
	public void start(StartContext context) throws StartException {
		if (!started.compareAndSet(false, true)) {
			throw new StartException("Service still active!");
		}
		LOGGER.info("Start HADeployService");

		File baseDir = env.getValue().getServerBaseDir();
		File haDeployDir = new File(baseDir, "ha-deployments");

		if (haDeployDir.exists()) {
			if (haDeployDir.isDirectory()) {
				List<ScannerTask> tasks = scanDirectory(haDeployDir);
				process(tasks);
				LOGGER.fine("Deployed "+this.deployed);
			} else {
				LOGGER.severe("HASingleton deployment: '" + haDeployDir.getPath() + "' is not a directory!");
			}
		} else {
			LOGGER.warning("No HASingleton deployment, directory does not exist : " + haDeployDir.getPath());
		}
	}

	private void process(List<ScannerTask> tasks) {
		final long timeout = 600;

		if(tasks.size() > 0) {
			List<ModelNode> updates = new ArrayList<ModelNode>(tasks.size());
			for (ScannerTask task : tasks) {
				final ModelNode update = task.getUpdate();
				if (LOGGER.isLoggable(Level.FINEST)) {
					LOGGER.finest("update action ["+ update + "]");
				}
				updates.add(update);
			}
			if(!updates.isEmpty()) {
				final ThreadFactory threadFactory = new JBossThreadFactory(new ThreadGroup("HADeploymentScanner-threads"), Boolean.FALSE, null, "%G - %t", null, null, AccessController.getContext());
				final ScheduledExecutorService executorService = Executors.newScheduledThreadPool(1, threadFactory);
				final Future<ModelNode> futureResults = deploy(getCompositeUpdate(updates), controllerValue.getValue().createClient(executorService), executorService);
				final ModelNode results;
				try {
					results = futureResults.get(timeout, TimeUnit.SECONDS);
					for (ScannerTask task : tasks) {
						task.handleSuccessResult();
					}
				} catch (TimeoutException e) {
					futureResults.cancel(true);
					final ModelNode failure = new ModelNode();
					failure.get(OUTCOME).set(FAILED);
					failure.get(FAILURE_DESCRIPTION).set("Did not receive a response to the deployment operation within the allowed timeout period [" + timeout +
							" seconds]. Check the server configuration file and the server logs to find more about the status of " +
							"the deployment.");
					for (ScannerTask task : tasks) {
						task.handleFailureResult(failure);
					}
				} catch (Exception e) {
					LOGGER.log(Level.SEVERE,"HASingleton deployment failed!",e);
					futureResults.cancel(true);
					final ModelNode failure = new ModelNode();
					failure.get(OUTCOME).set(FAILED);
					failure.get(FAILURE_DESCRIPTION).set(e.getMessage());
					for (ScannerTask task : tasks) {
						task.handleFailureResult(failure);
					}
				}
			}
		}
	}

	private List<ScannerTask> scanDirectory(File directory) {
		final ArrayList<ScannerTask> tasks = new ArrayList<ScannerTask>();
		LOGGER.finest("Start scanning '" + directory.getPath() + "'");

		File[] files = directory.listFiles();

		if (files == null) {
			LOGGER.fine("No HASingleton deployment files found");
		}

		for (File file : files) {
			final String fileName = file.getName();
			if ((fileName.endsWith(".ear") || fileName.endsWith(".jar") || fileName.endsWith(".war") || fileName.endsWith(".xml"))
         && file.isFile()) {
				LOGGER.fine("Start deploying " + fileName);
				tasks.add(new DeployTask(file.getAbsolutePath(), true,fileName, file));
			} else {
				LOGGER.finer("Ignore " + fileName + " for deploy");
			}
		}
		return tasks;
	}

	@Override
	public void stop(StopContext context) {
		if (started.get()) {
			LOGGER.info("Stop HADeployService");

			LOGGER.finest("Stopping deployments : "+this.deployed);

			final ArrayList<ScannerTask> tasks = new ArrayList<ScannerTask>();

			for (String deployment : this.deployed) {
				LOGGER.fine("Stop deployment " + deployment);
				tasks.add(new UndeployTask(deployment));
			}
			process(tasks);
			started.set(false);
		} else {
			LOGGER.warning("Service was not started!");
		}
	}

	public Future<ModelNode> deploy(final ModelNode operation, final ModelControllerClient modelController, final ScheduledExecutorService scheduledExecutor) {
		return scheduledExecutor.submit(new Callable<ModelNode>() {
			@Override
			public ModelNode call() throws Exception {
				return modelController.execute(operation);
			}
		});
	}

	private ModelNode getCompositeUpdate(final ModelNode... updates) {
		return getCompositeUpdate(Arrays.asList(updates));
	}
	private ModelNode getCompositeUpdate(final List<ModelNode> updates) {
		final ModelNode op = Util.getEmptyOperation(COMPOSITE, new ModelNode());
		final ModelNode steps = op.get(STEPS);
		for (ModelNode update : updates) {
			steps.add(update);
		}
		return op;
	}



	private abstract class ScannerTask {
		protected final String deploymentName;

		private ScannerTask(final String deploymentName) {
			this.deploymentName = deploymentName;
		}

		protected abstract ModelNode getUpdate();

		protected abstract void handleSuccessResult();

		protected abstract void handleFailureResult(final ModelNode result);
	}

	private abstract class ContentAddingTask extends ScannerTask {
		private final String path;
		private final boolean archive;

		protected ContentAddingTask(final String path, final boolean archive, final String deploymentName) {
			super(deploymentName);
			this.path = path;
			this.archive = archive;
		}

		protected ModelNode createContent() {
			final ModelNode content = new ModelNode();
			final ModelNode contentItem = content.get(0);
			contentItem.get(PATH).set(path);
			//			if (relativeTo != null) {
			//				contentItem.get(RELATIVE_TO).set(relativeTo);
			//			}
			contentItem.get(ARCHIVE).set(archive);
			return content;
		}

		@Override
		protected void handleSuccessResult() {
			LOGGER.finest("handleSuccess deploy of "+deploymentName);
			if (deployed.contains(deploymentName)) {
				deployed.remove(deploymentName);
			}
			deployed.add(deploymentName);
		}
	}

	private final class DeployTask extends ContentAddingTask {
		private DeployTask(final String path, final boolean archive, final String deploymentName, final File deploymentFile) {
			super(path, archive, deploymentName);
		}

		@Override
		protected ModelNode getUpdate() {
			final ModelNode address = new ModelNode().add(DEPLOYMENT, deploymentName);
			final ModelNode addOp = Util.getEmptyOperation(DeploymentAddHandler.OPERATION_NAME, address);
			addOp.get(CONTENT).set(createContent());
			addOp.get(PERSISTENT).set(false);
			final ModelNode deployOp = Util.getEmptyOperation(DeploymentDeployHandler.OPERATION_NAME, address);
			return getCompositeUpdate(addOp, deployOp);
		}

		@Override
		protected void handleFailureResult(final ModelNode result) {
			LOGGER.severe(result.get(FAILURE_DESCRIPTION).asString());
		}
	}

	private final class UndeployTask extends ScannerTask {

		private UndeployTask(final String deploymentName) {
			super(deploymentName);
		}

		@Override
		protected ModelNode getUpdate() {
			final ModelNode address = new ModelNode().add(DEPLOYMENT, deploymentName);
			final ModelNode undeployOp = Util.getEmptyOperation(DeploymentUndeployHandler.OPERATION_NAME, address);
			final ModelNode removeOp = Util.getEmptyOperation(DeploymentRemoveHandler.OPERATION_NAME, address);
			return getCompositeUpdate(undeployOp, removeOp);
		}

		@Override
		protected void handleSuccessResult() {
			deployed.remove(deploymentName);
		}

		@Override
		protected void handleFailureResult(ModelNode result) {
			LOGGER.warning("Undeploy failed");
			// TODO ModelNode auswerten
		}
	}
}
