package com.redhat.refarch.jdg.gasshopper;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

public class Coordinates implements Serializable
{

	private static final long serialVersionUID = 1L;
	private double latitude;
	private double longitude;
	private static final double R = 3959.87; // In miles

	public Coordinates(double latitude, double longitude)
	{
		this.latitude = latitude;
		this.longitude = longitude;
	}

	public Coordinates(String latitude, String longitude) throws NumberFormatException
	{
		this.latitude = Double.parseDouble( latitude );
		this.longitude = Double.parseDouble( longitude );
	}

	public double getLatitude()
	{
		return latitude;
	}

	public double getLongitude()
	{
		return longitude;
	}

	public double distance(Coordinates coordinates)
	{
		double dLat = Math.toRadians( coordinates.latitude - latitude );
		double dLon = Math.toRadians( coordinates.longitude - longitude );
		double lat1 = Math.toRadians( latitude );
		double lat2 = Math.toRadians( coordinates.latitude );

		double a = Math.sin( dLat / 2 ) * Math.sin( dLat / 2 ) + Math.sin( dLon / 2 ) * Math.sin( dLon / 2 ) * Math.cos( lat1 ) * Math.cos( lat2 );
		double c = 2 * Math.asin( Math.sqrt( a ) );
		return R * c;
	}

	public Coordinates calculateCellCenter()
	{
		double markerLatitude = Math.round( latitude );
		double markerLongitude = Math.round( longitude );
		return new Coordinates( markerLatitude, markerLongitude );
	}

	public Collection<Coordinates> calculateApplicableCellCenters(double radius)
	{
		Coordinates closestCellBorder = new Coordinates( Math.floor( latitude ) + 0.5, Math.floor( longitude ) + 0.5 );
		if( distance( closestCellBorder ) > radius )
		{
			//search area is a single cell
			Collection<Coordinates> cellCenters = new ArrayList<Coordinates>();
			cellCenters.add( calculateCellCenter() );
			return cellCenters;
		}
		else
		{
			Set<Coordinates> cellBorders = new HashSet<Coordinates>();
			double firstLatitude = closestCellBorder.latitude;
			double firstLongitude = closestCellBorder.longitude;
			for( int latitudeIndex = 0;; latitudeIndex++ )
			{
				int outerLoopSize = cellBorders.size();
				for( int longitudeIndex = 0;; longitudeIndex++ )
				{
					int innerLoopSize = cellBorders.size();
					addIfWithinRadius( cellBorders, new Coordinates( firstLatitude + latitudeIndex, firstLongitude + longitudeIndex ), radius );
					addIfWithinRadius( cellBorders, new Coordinates( firstLatitude + latitudeIndex, firstLongitude - longitudeIndex ), radius );
					addIfWithinRadius( cellBorders, new Coordinates( firstLatitude - latitudeIndex, firstLongitude + longitudeIndex ), radius );
					addIfWithinRadius( cellBorders, new Coordinates( firstLatitude - latitudeIndex, firstLongitude - longitudeIndex ), radius );
					if( cellBorders.size() == innerLoopSize )
					{
						break;
					}
				}
				if( cellBorders.size() == outerLoopSize )
				{
					break;
				}
			}
			Set<Coordinates> cellCenters = new HashSet<Coordinates>();
			for( Coordinates cellBorder : cellBorders )
			{
				cellCenters.add( new Coordinates( Math.floor( cellBorder.latitude ), Math.floor( cellBorder.longitude ) ) );
				cellCenters.add( new Coordinates( Math.floor( cellBorder.latitude ), Math.ceil( cellBorder.longitude ) ) );
				cellCenters.add( new Coordinates( Math.ceil( cellBorder.latitude ), Math.floor( cellBorder.longitude ) ) );
				cellCenters.add( new Coordinates( Math.ceil( cellBorder.latitude ), Math.ceil( cellBorder.longitude ) ) );
			}
			return cellCenters;
		}
	}

	private void addIfWithinRadius(Set<Coordinates> cellBorders, Coordinates candidate, double radius)
	{
		if( distance( candidate ) <= radius )
		{
			cellBorders.add( candidate );
		}
	}

	public String asString()
	{
		return "(" + latitude + "," + longitude + ")";
	}

	public static Coordinates fromString(String coordinateString)
	{
		int separatorIndex = coordinateString.indexOf( ',' );
		String latitude = coordinateString.substring( 1, separatorIndex );
		String longitude = coordinateString.substring( separatorIndex + 1, coordinateString.indexOf( ')' ) );
		Coordinates coordinates = new Coordinates( latitude, longitude );
		return coordinates;
	}

	@Override
	public String toString()
	{
		return "Coordinates [latitude=" + latitude + ", longitude=" + longitude + "]";
	}
}