package com.redhat.refarch.jdg.gasshopper;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringWriter;
import java.lang.reflect.Type;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import org.infinispan.client.hotrod.RemoteCache;
import org.infinispan.client.hotrod.RemoteCacheManager;
import org.infinispan.client.hotrod.configuration.ConfigurationBuilder;
import org.infinispan.commons.util.Base64;

import au.com.bytecode.opencsv.CSVReader;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonParser;

public class Client
{

	public static void main(String[] args) throws Exception
	{
		Client loader = new Client();
		if( args.length > 0 )
		{
			if( args[0].equals( "load" ) )
			{
				Collection<String> gasStations = loader.cacheGasStations( args[1] );
				loader.cachePriceReports( gasStations );
			}
			else if( args[0].equals( "stations" ) )
			{
				String latitude = args[1];
				String longitude = args[2];
				Coordinates coordinates = new Coordinates( latitude, longitude );
				String username = args[3];
				String password = args[4];
				getGasStationsREST( coordinates, username, password );
			}
			else if( args[0].equals( "prices" ) )
			{
				String uuid = args[1];
				String username = args[2];
				String password = args[3];
				getPriceReportREST( uuid, username, password );
			}
			else if( args[0].equals( "size" ) )
			{
				String[] caches = new String[] {"stations", "prices"};
				for( String cache : caches )
				{
					RemoteCache<String, Object> remoteCache = getCache( cache );
					int nodeSize = remoteCache.size();
					int totalSize = remoteCache.keySet().size();
					System.out.println( cache + " cache size is " + totalSize + " and " + nodeSize
							+ " items are stored directly on the given JDG node" );
				}
			}
			else if( args[0].equals( "size2" ) )
			{
				String[] caches = new String[] {"stations", "prices"};
				for( String cache : caches )
				{
					RemoteCache<String, Object> remoteCache = getCache( cache );
					int nodeSize = remoteCache.size();
					System.out.println( cache + " has " + nodeSize + " items stored directly on the given JDG node" );
				}
			}
			else
			{
				printHelp();
			}
		}
		else
		{
			printHelp();
		}
	}

	private static void printHelp()
	{
		System.out.println( "Provide appropriate arguments as follows:" );
		System.out.println();
		System.out.println( "To load gas station date from files and generate mockup price reports:" );
		System.out.println( "	load file1.csv,file2.csv,file3.csv..." );
		System.out.println();
		System.out.println( "To query all gas stations within identified grid cell through REST:" );
		System.out.println( "	stations latitude longitude username password" );
		System.out.println();
		System.out
				.println( "To get price reports through REST, most recent priceReportId is always stationId. Next id will be in the returned report:" );
		System.out.println( "	prices priceReportId username password" );
		System.out.println();
		System.out.println( "For information on the size of each cache:" );
		System.out.println( "	size" );
	}

	private static Map<String, PriceReport> generatePriceReports(Collection<String> stationIds, int reportPerStation, float basePrice)
	{
		Date[] reportDates = new Date[reportPerStation];
		long interval = 12 * 60 * 60000 / ( reportPerStation - 1 );
		long millis = System.currentTimeMillis() - 12 * 60 * 60000;
		for( int index = 0; index < reportDates.length; index++ )
		{
			reportDates[index] = new Date( millis );
			millis += interval;
		}

		Map<String, PriceReport> priceReports = new HashMap<String, PriceReport>();
		for( String stationId : stationIds )
		{
			for( int index = 0; index < reportDates.length; index++ )
			{
				PriceReport priceReport = PriceReportGenerator.getRandomPriceReport( basePrice, 1.20f );
				priceReport.setDate( reportDates[index] );
				if( index > 0 )
				{
					priceReport.setLastReportKey( stationId + "_" + reportDates[index - 1].getTime() );
				}
				String key;
				if( index == reportDates.length - 1 )
				{
					key = stationId;
				}
				else
				{
					key = stationId + "_" + reportDates[index].getTime();
				}
				priceReports.put( key, priceReport );
			}
		}
		return priceReports;
	}

	private static void getGasStationsREST(Coordinates coordinates, String username, String password) throws IOException
	{
		Properties properties = new Properties();
		properties.load( Client.class.getResourceAsStream( "/jdg.properties" ) );
		String host = properties.getProperty( "jdg.host" );
		host = System.getProperty( "jdg.host", host );
		String urlServerAddress = "http://" + host + ":8080/rest/stations/" + coordinates.asString();
		String response = restGet( urlServerAddress, username, password );

		Map<UUID, GasStation> gasStations = new HashMap<UUID, GasStation>();
		JsonObject jsonObject = new JsonParser().parse( response ).getAsJsonObject();
		for( Entry<String, JsonElement> jsonElementEntry : jsonObject.entrySet() )
		{
			JsonElement jsonElement = jsonElementEntry.getValue();
			GasStation gasStation = (GasStation)new Gson().fromJson( jsonElement, GasStation.class );
			gasStations.put( gasStation.getStationId(), gasStation );
		}

		System.out.println( "REST response parsed to " + gasStations.size() + " gas stations, here is a random example:" );
		System.out.println( gasStations.values().iterator().next() );
		System.out.println( "\n\nAll station UUIDs:\n**************************" );
		for( GasStation gasStation : gasStations.values() )
		{
			System.out.println( gasStation.getStationId() );
		}
		System.out.println( "**************************\n" );
	}

	private static void getPriceReportREST(String uuid, String username, String password) throws IOException
	{
		Properties properties = new Properties();
		properties.load( Client.class.getResourceAsStream( "/jdg.properties" ) );
		String host = properties.getProperty( "jdg.host" );
		host = System.getProperty( "jdg.host", host );
		String urlServerAddress = "http://" + host + ":8080/rest/prices/" + uuid;
		String response = restGet( urlServerAddress, username, password );
		System.out.println( response );
		JsonDeserializer<Date> dateDeserializer = new JsonDeserializer<Date>()
		{

			@Override
			public Date deserialize(JsonElement jsonElement, Type type, JsonDeserializationContext context) throws JsonParseException
			{
				return new Date( jsonElement.getAsLong() );
			}
		};
		GsonBuilder gsonBuilder = new GsonBuilder();
		gsonBuilder.registerTypeAdapter( Date.class, dateDeserializer );
		Gson gson = gsonBuilder.create();
		PriceReport priceReport = (PriceReport)gson.fromJson( response, PriceReport.class );
		System.out.println( "\n\nREST response parsed to PriceReport object:" );
		System.out.println( "**************************" );
		System.out.println( priceReport );
		System.out.println( "**************************" );
	}

	private static String restGet(String urlServerAddress, String username, String password) throws IOException
	{
		System.out.println( "----------------------------------------" );
		System.out.println( "Executing GET " + urlServerAddress );
		System.out.println( "----------------------------------------" );

		HttpURLConnection connection = (HttpURLConnection)new URL( urlServerAddress ).openConnection();
		connection.setRequestProperty( "Authorization", "Basic " + Base64.encodeBytes( ( username + ":" + password ).getBytes() ) );
		connection.setRequestMethod( "GET" );
		connection.setRequestProperty( "Content-Type", "text/plain" );
		connection.setRequestProperty( "Accept", "application/json" );
		connection.setDoOutput( true );
		connection.connect();

		Reader responseReader = new InputStreamReader( connection.getInputStream() );
		StringWriter stringWriter = new StringWriter();
		char[] buffer = new char[4096];
		int n = 0;
		while( -1 != ( n = responseReader.read( buffer ) ) )
		{
			stringWriter.write( buffer, 0, n );
		}
		String response = stringWriter.toString();

		System.out.println( "----------------------------------------" );
		System.out.println( connection.getResponseCode() + " " + connection.getResponseMessage() );
		System.out.println( "----------------------------------------" );
		connection.disconnect();

		return response;
	}

	private Collection<String> cacheGasStations(String filenames) throws Exception
	{
		Map<String, Map<UUID, GasStation>> gasStations = readGasStations( filenames );
		System.out.println( "Will store gas station info as " + gasStations.size() + " cache entries" );
		getCache( "stations" ).putAll( gasStations, -1, TimeUnit.DAYS );
		Collection<String> stationIds = new ArrayList<String>();
		for( Map<UUID, GasStation> map : gasStations.values() )
		{
			Set<UUID> keys = map.keySet();
			for( UUID uuid : keys )
			{
				stationIds.add( uuid.toString() );
			}
		}
		return stationIds;
	}

	private void cachePriceReports(Collection<String> stationIds) throws Exception
	{
		Map<String, PriceReport> priceReports = generatePriceReports( stationIds, 10, 4.19f );
		System.out.println( "Will store price reports as " + priceReports.size() + " cache entries" );
		getCache( "prices" ).putAll( priceReports, 1, TimeUnit.DAYS );
	}

	private static RemoteCache<String, Object> getCache(String cacheName) throws IOException
	{
		Properties properties = new Properties();
		properties.load( Client.class.getResourceAsStream( "/jdg.properties" ) );
		ConfigurationBuilder builder = new ConfigurationBuilder();
		String host = properties.getProperty( "jdg.host" );
		String port = properties.getProperty( "jdg.port" );
		//System properties override the defaults in our property files:
		host = System.getProperty( "jdg.host", host );
		port = System.getProperty( "jdg.port", port );
		builder.addServer().host( host ).port( Integer.valueOf( port ) );
		RemoteCacheManager cacheManager = new RemoteCacheManager( builder.build() );
		RemoteCache<String, Object> cache = cacheManager.getCache( cacheName );
		return cache;
	}

	private Map<String, Map<UUID, GasStation>> readGasStations(String filenames) throws Exception
	{
		List<String[]> stations = new ArrayList<String[]>();
		String[] csvFiles = filenames.split( "," );
		for( String csvFile : csvFiles )
		{
			CSVReader csvReader = new CSVReader( new InputStreamReader( new FileInputStream( csvFile ) ) );
			stations.addAll( csvReader.readAll() );
			csvReader.close();
		}
		System.out.println( "Read " + stations.size() + " stations from file" );

		Map<String, Map<UUID, GasStation>> gasStations = new HashMap<String, Map<UUID, GasStation>>();
		for( String[] stationDetails : stations )
		{
			Coordinates coordinates = new Coordinates( stationDetails[1], stationDetails[0] );
			String[] addressParts = stationDetails[3].split( "\n" );
			String address = addressParts[0] + ", " + addressParts[1];
			String phone = addressParts[2];
			GasStation gasStation = new GasStation( stationDetails[2], address, phone, coordinates );
			Map<UUID, GasStation> cellStations = gasStations.get( coordinates.calculateCellCenter().asString() );
			if( cellStations == null )
			{
				cellStations = new HashMap<UUID, GasStation>();
				gasStations.put( coordinates.calculateCellCenter().asString(), cellStations );
			}
			cellStations.put( gasStation.getStationId(), gasStation );
		}

		return gasStations;
	}
}
